import json
import pytz

from datetime import datetime
from timestream.dimensions import TimestreamDimensions
from timestream.util import is_time_key, is_version_key


class TimestreamRecord:
    '''
    A TimestreamRecord represents a single record that can be loaded into Timestream.
    TimestreamRecords are immutable (i.e. after their creation they cannot be updated).
    This immutability is intended to preserve the validity of the TimestreamRecord's
    attributes (which are validated on init)
    '''

    def __init__(
        self,
        measure_name: str,
        measure_value: float,
        dimensions: TimestreamDimensions = None,
        mstime: int = None
    ):
        assert measure_name and type(measure_name) is str, f'Invalid measure name "{measure_name}" with type "{type(measure_name)}"'
        assert not is_time_key(measure_name), f'Invalid measure name "{measure_name}". Keyword "time" cannot be used as a measure name'
        assert not is_version_key(measure_name), f'Invalid measure name "{measure_name}". Keyword "version" cannot be used as a measure name'
        assert type(measure_value) in [int, float], f'Invalid measure value type "{type(measure_value)}"'
        assert not dimensions or type(dimensions) is TimestreamDimensions, f'Invalid dimensions for TimestreamRecord with type {type(dimensions)}'
        assert not mstime or (type(mstime) is int and mstime > 0), f'Millisecond time must be an integer greater than 0, got {mstime}'

        # Immutable fields
        self.__measure_name = str(measure_name).strip()
        self.__measure_value = float(measure_value)
        self.__dimensions = dimensions or TimestreamDimensions()
        self.__mstime = None if mstime is None else int(mstime)

    @property
    def measure_name(self) -> str:
        return self.__measure_name

    @property
    def measure_value(self) -> float:
        return self.__measure_value

    @property
    def dimensions(self) -> TimestreamDimensions:
        return self.__dimensions.copy()

    @property
    def mstime(self) -> int:
        return self.__mstime

    @property
    def time(self) -> datetime:
        '''Return the mstime as a UTC datetime'''
        if self.mstime:
            # fromtimestamp returns a naive datetime object that IS adjusted for the local timezone
            return datetime.fromtimestamp(self.mstime/1000).astimezone().astimezone(pytz.utc)
        return None

    def loadable(self, include_dimensions: bool = True, omit_dimensions: TimestreamDimensions = None) -> dict:
        '''loadable returns a dictionary that is ready to load into Timestream'''
        loadable = {
            'MeasureName': self.measure_name,
            'MeasureValue': str(self.measure_value).strip()
        }
        if self.mstime is not None:
            loadable['Time'] = str(self.mstime).strip()
        if include_dimensions:
            loadable['Dimensions'] = self.__dimensions.loadable(
                omit_dimensions=omit_dimensions.get_dimension_keys()
            )
        return loadable

    def to_dict(self) -> dict:
        '''to_dict returns a human friendly dictionary representation of this TimestreamRecord'''
        record_dict = {
            'measure_name': self.measure_name,
            'measure_value::double': self.measure_value
        }
        if self.time is not None:
            # format the time the way Timestream formats time
            record_dict['time'] = self.time.strftime('%Y-%m-%d %H:%M:%S.%f000')
        record_dict = {**record_dict, **self.dimensions.to_dict()}
        return record_dict

    def __repr__(self) -> str:
        return json.dumps(self.to_dict())


class RejectedTimestreamRecord:
    '''
    RejectedTimestreamRecord is an immutable class representing a TimestreamRecord that got rejected by Timestream during load.
    This class carries the error message that Timestream returned when rejecting this record.
    '''

    def __init__(self, record: TimestreamRecord, error: Exception):
        assert type(record) is TimestreamRecord
        assert error is not None
        self.__record = record
        self.__error = error

    @property
    def record(self) -> TimestreamRecord:
        return self.__record

    @property
    def error(self) -> Exception:
        return self.__error

    def to_dict(self) -> dict:
        return {
            'record': self.record.to_dict(),
            'error': str(self.error)
        }

    def __repr__(self) -> str:
        return json.dumps(self.to_dict())
