import pytest
from unittest.mock import Mock

from poe_new_relic.experimental_client import ExperimentalClient, TooManySessionsError
from test_utils.mock import MockApi


class TestExperimentalClient_UnitTests:

    @pytest.mark.unit
    def test_enter_successful_login_without_expected_cookie(self):
        """Test successful login but without expected cookie (should log warning)."""
        username = "test@example.com"
        password = "password123"

        # Mock successful login response without expected cookie
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.cookies = {}
        setattr(mock_response, "next", None)

        mock_session = MockApi()
        mock_session.headers = {}
        mock_session.on(
            "post",
            "https://login.newrelic.com/login",
            data={"login[email]": username, "login[password]": password},
            allow_redirects=False,
        ).return_val(mock_response)

        mock_logger = Mock()
        client = ExperimentalClient(username, password, session=mock_session, logger=mock_logger)

        result = client.__enter__()

        assert result == client
        mock_logger.warning.assert_called_once()
        mock_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_enter_too_many_sessions_error(self):
        """Test login failure due to too many active sessions."""
        username = "test@example.com"
        password = "password123"

        # Mock response that redirects to limit_active_sessions
        mock_next = Mock()
        mock_next.url = "https://login.newrelic.com/limit_active_sessions"

        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        setattr(mock_response, "next", mock_next)

        mock_session = MockApi()
        mock_session.headers = {}
        mock_session.on(
            "post",
            "https://login.newrelic.com/login",
            data={"login[email]": username, "login[password]": password},
            allow_redirects=False,
        ).return_val(mock_response)

        client = ExperimentalClient(username, password, session=mock_session)

        with pytest.raises(TooManySessionsError) as exc_info:
            client.__enter__()

        assert username in str(exc_info.value)
        mock_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_decline_user_upgrade_request_success(self):
        """Test successful decline of user upgrade request."""
        username = "admin@example.com"
        password = "password123"
        request_id = "310355"

        expected_response = {
            "data": {
                "reviewPendingRequest": {
                    "request": {
                        "id": request_id,
                        "status": "DENIED",
                        "requestType": "TIER_UPGRADE",
                        "requestMessage": None,
                    }
                }
            }
        }

        expected_mutation = {
            "query": f"""
mutation {{
  reviewPendingRequest(
    input: {{
      id: "{request_id}",
      status: DENIED,
      reviewMessage: "New Relic access is managed through Active Directory and myAccess.\nPlease request the appropriate role for the requested access.\nYou can learn more about it here: https://pearsoneducationinc.sharepoint.com/sites/GRPTOToolsSupport-SETs-/SitePages/New-Relic-User-Access-Request---myAccess.aspx"
    }}
  ) {{
    request {{
      id
      status
      requestType
      requestMessage
    }}
    errors {{
      message
    }}
  }}
}}
        """
        }

        # Mock successful response
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_session = MockApi()
        mock_session.headers = {}
        mock_session.on(
            "post",
            "https://user-graphql.service.newrelic.com/public/graphql",
            json=expected_mutation,
        ).return_val(mock_response)

        client = ExperimentalClient(username, password, session=mock_session)

        result = client.decline_user_upgrade_request(request_id)

        expected_result = {
            "id": request_id,
            "status": "DENIED",
            "requestType": "TIER_UPGRADE",
            "requestMessage": None,
        }

        assert result == expected_result
        mock_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_decline_user_upgrade_request_api_error(self):
        """Test decline user upgrade request with API errors in response."""
        username = "admin@example.com"
        password = "password123"
        request_id = "310355"

        expected_error_response = {"errors": [{"message": "Request not found"}]}

        expected_mutation = {
            "query": f"""
mutation {{
  reviewPendingRequest(
    input: {{
      id: "{request_id}",
      status: DENIED,
      reviewMessage: "New Relic access is managed through Active Directory and myAccess.\nPlease request the appropriate role for the requested access.\nYou can learn more about it here: https://pearsoneducationinc.sharepoint.com/sites/GRPTOToolsSupport-SETs-/SitePages/New-Relic-User-Access-Request---myAccess.aspx"
    }}
  ) {{
    request {{
      id
      status
      requestType
      requestMessage
    }}
    errors {{
      message
    }}
  }}
}}
        """
        }

        # Mock error response
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_error_response)

        mock_session = MockApi()
        mock_session.headers = {}
        mock_session.on(
            "post",
            "https://user-graphql.service.newrelic.com/public/graphql",
            json=expected_mutation,
        ).return_val(mock_response)

        client = ExperimentalClient(username, password, session=mock_session)

        with pytest.raises(Exception) as exc_info:
            client.decline_user_upgrade_request(request_id)

        assert "Error declining user upgrade request:" in str(exc_info.value)
        mock_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_context_manager_full_workflow(self):
        """Test complete context manager workflow with login and logout."""
        username = "test@example.com"
        password = "password123"

        # Mock login response
        mock_login_response = MockApi()
        mock_login_response.on("raise_for_status").return_val(None)
        mock_login_response.cookies = {"login_service_login_newrelic_com_tokens": "test_token"}
        setattr(mock_login_response, "next", None)

        # Mock logout response
        mock_logout_response = MockApi()
        mock_logout_response.on("raise_for_status").return_val(None)

        mock_session = MockApi()
        mock_session.headers = {}

        # Setup login mock
        mock_session.on(
            "post",
            "https://login.newrelic.com/login",
            data={"login[email]": username, "login[password]": password},
            allow_redirects=False,
        ).return_val(mock_login_response)

        # Setup logout mock
        mock_session.on(
            "get",
            "https://login.newrelic.com/logout",
            allow_redirects=False,
        ).return_val(mock_logout_response)

        client = ExperimentalClient(username, password, session=mock_session)

        with client as c:
            assert c == client

        mock_session.assert_expectations()
        mock_login_response.assert_expectations()
        mock_logout_response.assert_expectations()

    @pytest.mark.unit
    def test_too_many_sessions_error_message(self):
        """Test TooManySessionsError exception message format."""
        username = "test@example.com"

        error = TooManySessionsError(username)

        expected_message = f"Too many active sessions for user {username}. You must manually sign in as {username} and end some sessions before using this client."
        assert str(error) == expected_message
