import pytest
import json

from poe_new_relic.api import NewRelic, NEWRELIC_GRAPHQL_URL
from test_utils.mock import MockApi
from typing import Tuple, Any, Optional


class TestNewRelic_UnitTests:
    @pytest.mark.unit
    def test_nerdgraph_nrql_search(self):
        # Set expected values
        expected_api_key = "derp"
        expected_account_id = 12345
        expected_nrql = "SELECT uniques(appName) FROM Transactions SINCE 1 hour ago"
        expected_label = "apmAppNames"
        expected_timeout_ms = 123456
        expected_results = {expected_label: {"results": ["Example App 1", "Example App 2"]}}
        expected_resp = {"data": {"actor": {"account": expected_results}}}

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_resp)

        mock_http = MockApi()
        mock_http.on(
            "post",
            url=NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json={
                "query": f'{{actor {{account(id: {expected_account_id}) {{apmAppNames: nrql(query: "{expected_nrql}") {{results}}}}}}}}',
            },
            timeout=expected_timeout_ms,
        ).return_val(mock_response)

        # Test nerdgraph_nrql_search()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        results = nr.nerdgraph_nrql_search(
            account_id=expected_account_id,
            nrql=expected_nrql,
            label=expected_label,
            timeout_ms=expected_timeout_ms,
        )
        print(f"Results: {json.dumps(results, indent=2)}")
        assert results == expected_results
        mock_http.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_nerdgraph_entitysearch(self):
        expected_api_key = "derp"
        expected_query = "sql"
        expected_output_keys = "{keys}"
        expected_on = "... on SyntheticMonitorEntityOutline"
        expected_return_list = [
            {"name": "derp Highered", "tags": [{"key": "wallBoardProducts", "values": ["durp"]}]}
        ]

        # Set up mocks
        mock_http, mock_response = self.get_mocks(expected_query, expected_output_keys, expected_on)

        # Test nerdgraph_entitysearch()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        resp = nr.nerdgraph_entitysearch(expected_query, expected_output_keys, on=expected_on)
        assert resp == expected_return_list
        mock_http.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_nerdgraph_entitysearch_exception(self):
        # Set expected values
        expected_api_key = "derp"
        expected_query = "sql"
        expected_output_keys = "{keys}"
        expected_on = "... on SyntheticMonitorEntityOutline"
        expected_exception = "test error"

        # Set up mocks
        mock_http, mock_response = self.get_mocks(
            expected_query, expected_output_keys, expected_on, is_error=1
        )

        # Test nerdgraph_entitysearch()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        with pytest.raises(Exception) as e_info:
            nr.nerdgraph_entitysearch(expected_query, expected_output_keys, on=expected_on)
        assert str(e_info.value) == expected_exception
        mock_http.assert_expectations()

    @pytest.mark.unit
    def test_get_synths(self):
        # Set expected values
        expected_api_key = "derp"
        expected_query = (
            'entitySearch(query: "domain = \u0027SYNTH\u0027 AND type = \u0027MONITOR\u0027")'
        )
        expected_output_keys = "{guid name tags {key values} monitorId}"
        expected_on = "... on SyntheticMonitorEntityOutline"
        expected_return_list = [
            {"name": "derp Highered", "tags": [{"key": "wallBoardProducts", "values": ["durp"]}]}
        ]

        # Set up mocks
        mock_http, mock_response = self.get_mocks(expected_query, expected_output_keys, expected_on)

        # Test get_synths()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        resp = nr.get_synths()
        assert resp == expected_return_list
        mock_http.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_get_monitorid_and_guids_synths_by_tag(self):
        # Set expected values
        expected_tag = "wallBoardProducts"
        expected_api_key = "derp"
        expected_query = (
            'entitySearch(query: "domain = \u0027SYNTH\u0027 AND type = \u0027MONITOR\u0027")'
        )
        expected_output_keys = "{guid name tags {key values} monitorId}"
        expected_return_list = {"durp": [{"guid": [None]}, {"monitorId": [None]}]}
        expected_on = "... on SyntheticMonitorEntityOutline"

        # Set up mocks
        mock_http, mock_response = self.get_mocks(expected_query, expected_output_keys, expected_on)

        # Test get_monitorid_and_guids_synths_by_tag()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        resp = nr.get_monitorid_and_guids_synths_by_tag(expected_tag)
        # the order of the keys can be flip flopped randomly, this covers that
        assert resp == expected_return_list or {"durp": [{"monitorId": [None]}, {"guid": [None]}]}
        mock_http.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_get_scorecard_performance_dashboards(self):
        # Set expected values
        expected_tag = "wallBoardProducts"
        expected_api_key = "derp"
        expected_query = 'entitySearch(query: "((name LIKE \u0027Scorecard -%\u0027) OR (name LIKE \u0027Performance -%\u0027)) AND (domain = \u0027VIZ\u0027 AND type = \u0027DASHBOARD\u0027)")'
        expected_output_keys = "{guid name updatedAt accountId permalink createdAt}"
        expected_return_list = [
            {"name": "derp Highered", "tags": [{"key": "wallBoardProducts", "values": ["durp"]}]}
        ]
        expected_on = "... on DashboardEntityOutline"

        # Set up mocks
        mock_http, mock_response = self.get_mocks(expected_query, expected_output_keys, expected_on)

        # Test get_scorecard_performance_dashboards()
        nr = NewRelic(expected_api_key, http_session=mock_http)
        resp = nr.get_scorecard_performance_dashboards()
        assert resp == expected_return_list
        mock_http.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_standard(self):
        """Test creating a standard account."""
        expected_api_key = "test_api_key"
        expected_name = "MyApp"
        expected_environment = "prod"
        expected_lob_suffix = "gptx"
        expected_account_type = "standard"
        expected_org_id = "12345"
        expected_region = "us01"
        expected_account_name = f"{expected_name}-{expected_environment}_{expected_lob_suffix}"

        # Expected response
        expected_response = {
            "data": {
                "accountManagementCreateAccount": {
                    "managedAccount": {"name": expected_account_name, "id": 67890}
                }
            }
        }

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_response)

        # Test create_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.create_account(
            name=expected_name,
            environment=expected_environment,
            lob_suffix=expected_lob_suffix,
            account_type=expected_account_type,
            org_id=expected_org_id,
            region=expected_region,
        )

        expected_result = {"name": expected_account_name, "id": 67890}
        assert result == expected_result
        mock_http_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_private(self):
        """Test creating a private account."""
        expected_api_key = "test_api_key"
        expected_name = "PrivateApp"
        expected_environment = "dev"
        expected_lob_suffix = "strg"
        expected_account_type = "private"
        expected_org_id = "67890"
        expected_region = "eu01"
        expected_account_name = (
            f"{expected_name}_SECURE-{expected_environment}_{expected_lob_suffix}"
        )

        # Expected response
        expected_response = {
            "data": {
                "accountManagementCreateAccount": {
                    "managedAccount": {"name": expected_account_name, "id": 54321}
                }
            }
        }

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_response)

        # Test create_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.create_account(
            name=expected_name,
            environment=expected_environment,
            lob_suffix=expected_lob_suffix,
            account_type=expected_account_type,
            org_id=expected_org_id,
            region=expected_region,
        )

        expected_result = {"name": expected_account_name, "id": 54321}
        assert result == expected_result
        mock_http_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_hsm(self):
        """Test creating an HSM account."""
        expected_api_key = "test_api_key"
        expected_name = "HSMApp"
        expected_environment = "prod"
        expected_lob_suffix = "asmq"
        expected_account_type = "hsm"
        expected_org_id = "11111"
        expected_region = "us01"
        expected_account_name = f"{expected_name}_HSM-{expected_environment}_{expected_lob_suffix}"

        # Expected response
        expected_response = {
            "data": {
                "accountManagementCreateAccount": {
                    "managedAccount": {"name": expected_account_name, "id": 98765}
                }
            }
        }

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_response)

        # Test create_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.create_account(
            name=expected_name,
            environment=expected_environment,
            lob_suffix=expected_lob_suffix,
            account_type=expected_account_type,
            org_id=expected_org_id,
            region=expected_region,
        )

        expected_result = {"name": expected_account_name, "id": 98765}
        assert result == expected_result
        mock_http_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_fedramp(self):
        """Test creating a FedRAMP account."""
        expected_api_key = "test_api_key"
        expected_name = "FedRAMPApp"
        expected_environment = "prod"
        expected_lob_suffix = "vlxx"
        expected_account_type = "fedramp"
        expected_org_id = "22222"
        expected_region = "us01"
        expected_account_name = f"{expected_name}_FR-{expected_environment}_{expected_lob_suffix}"

        # Expected response
        expected_response = {
            "data": {
                "accountManagementCreateAccount": {
                    "managedAccount": {"name": expected_account_name, "id": 13579}
                }
            }
        }

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_response)

        # Test create_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.create_account(
            name=expected_name,
            environment=expected_environment,
            lob_suffix=expected_lob_suffix,
            account_type=expected_account_type,
            org_id=expected_org_id,
            region=expected_region,
        )

        expected_result = {"name": expected_account_name, "id": 13579}
        assert result == expected_result
        mock_http_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_default_region(self):
        """Test creating an account with default region (us01)."""
        expected_api_key = "test_api_key"
        expected_name = "DefaultRegionApp"
        expected_environment = "staging"
        expected_lob_suffix = "asmq"
        expected_account_type = "standard"
        expected_org_id = "99999"
        expected_region = "us01"  # Default region
        expected_account_name = f"{expected_name}-{expected_environment}_{expected_lob_suffix}"

        # Expected response
        expected_response = {
            "data": {
                "accountManagementCreateAccount": {
                    "managedAccount": {"name": expected_account_name, "id": 11111}
                }
            }
        }

        # Set up mocks
        mock_response = MockApi()
        mock_response.on("raise_for_status").return_val(None)
        mock_response.on("json").return_val(expected_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_response)

        # Test create_account() without specifying region (should default to us01)
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.create_account(
            name=expected_name,
            environment=expected_environment,
            lob_suffix=expected_lob_suffix,
            account_type=expected_account_type,
            org_id=expected_org_id,
            # region not specified, should default to "us01"
        )

        expected_result = {"name": expected_account_name, "id": 11111}
        assert result == expected_result
        mock_http_session.assert_expectations()
        mock_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_invalid_lob_suffix(self):
        """Test error handling for invalid LOB suffix."""
        expected_api_key = "test_api_key"
        expected_name = "TestApp"
        expected_environment = "test"
        expected_lob_suffix = "invalid"  # Invalid LOB suffix
        expected_account_type = "standard"
        expected_org_id = "12345"

        # Test create_account() with invalid LOB suffix
        nr = NewRelic(expected_api_key)
        with pytest.raises(AssertionError) as e_info:
            nr.create_account(
                name=expected_name,
                environment=expected_environment,
                lob_suffix=expected_lob_suffix,
                account_type=expected_account_type,
                org_id=expected_org_id,
            )

        assert "Invalid lob_suffix. Must be one of" in str(e_info.value)

    @pytest.mark.unit
    def test_create_account_invalid_region(self):
        """Test error handling for invalid region."""
        expected_api_key = "test_api_key"
        expected_name = "TestApp"
        expected_environment = "test"
        expected_lob_suffix = "gptx"
        expected_account_type = "standard"
        expected_org_id = "12345"
        expected_region = "invalid"  # Invalid region

        # Test create_account() with invalid region
        nr = NewRelic(expected_api_key)
        with pytest.raises(AssertionError) as e_info:
            nr.create_account(
                name=expected_name,
                environment=expected_environment,
                lob_suffix=expected_lob_suffix,
                account_type=expected_account_type,
                org_id=expected_org_id,
                region=expected_region,
            )

        assert "Invalid region. Must be one of" in str(e_info.value)

    @pytest.mark.unit
    def test_create_account_invalid_account_type(self):
        """Test error handling for invalid account type."""
        expected_api_key = "test_api_key"
        expected_name = "TestApp"
        expected_environment = "test"
        expected_lob_suffix = "gptx"
        expected_account_type = "invalid"  # Invalid account type
        expected_org_id = "12345"

        # Test create_account() with invalid account type
        nr = NewRelic(expected_api_key)
        with pytest.raises(AssertionError) as e_info:
            nr.create_account(
                name=expected_name,
                environment=expected_environment,
                lob_suffix=expected_lob_suffix,
                account_type=expected_account_type,
                org_id=expected_org_id,
            )

        assert "Invalid account_type. Must be one of" in str(e_info.value)

    @pytest.mark.unit
    def test_create_account_api_error(self):
        """Test error handling when API returns errors."""
        expected_api_key = "test_api_key"
        expected_name = "ErrorApp"
        expected_environment = "test"
        expected_lob_suffix = "ellx"
        expected_account_type = "standard"
        expected_org_id = "12345"
        expected_region = "us01"
        expected_account_name = f"{expected_name}-{expected_environment}_{expected_lob_suffix}"

        # Expected response with errors
        expected_error_response = {
            "errors": [{"message": "Insufficient permissions to create account"}]
        }

        # Set up mocks
        mock_error_response = MockApi()
        mock_error_response.on("raise_for_status").return_val(None)
        mock_error_response.on("json").return_val(expected_error_response)

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_error_response)

        # Test create_account() with API error
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        with pytest.raises(Exception) as e_info:
            nr.create_account(
                name=expected_name,
                environment=expected_environment,
                lob_suffix=expected_lob_suffix,
                account_type=expected_account_type,
                org_id=expected_org_id,
                region=expected_region,
            )

        assert "Error while submitting account creation request:" in str(e_info.value)
        mock_http_session.assert_expectations()
        mock_error_response.assert_expectations()

    @pytest.mark.unit
    def test_create_account_http_error(self):
        """Test error handling when HTTP request fails."""
        expected_api_key = "test_api_key"
        expected_name = "HTTPErrorApp"
        expected_environment = "test"
        expected_lob_suffix = "vlxx"
        expected_account_type = "standard"
        expected_org_id = "12345"
        expected_region = "us01"
        expected_account_name = f"{expected_name}-{expected_environment}_{expected_lob_suffix}"

        # Set up mocks
        mock_error_response = MockApi()
        mock_error_response.on("raise_for_status").raise_exception(
            Exception("HTTP 401: Unauthorized")
        )

        mock_http_session = MockApi()

        # Expected mutation
        expected_mutation = {
            "query": f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{expected_region}",
            organizationId: "{expected_org_id}",
            name: "{expected_account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""
        }

        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_mutation,
        ).return_val(mock_error_response)

        # Test create_account() with HTTP error
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        with pytest.raises(Exception) as e_info:
            nr.create_account(
                name=expected_name,
                environment=expected_environment,
                lob_suffix=expected_lob_suffix,
                account_type=expected_account_type,
                org_id=expected_org_id,
                region=expected_region,
            )

        assert "HTTP 401: Unauthorized" in str(e_info.value)
        mock_http_session.assert_expectations()
        mock_error_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_standard_basic(self):
        """Test adding a StandardBasic group to an account."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "WW New Relic Users sBasic_myapp-prod"
        expected_org_id = "67890"
        expected_group_id = "group-123-456"

        # Expected group lookup response
        expected_group_lookup_response = {
            "data": {
                "customerAdministration": {
                    "groups": {"items": [{"id": expected_group_id, "name": expected_group_name}]}
                }
            }
        }

        # Expected role assignment response
        expected_role_response = {
            "data": {
                "authorizationManagementGrantAccess": {
                    "roles": [
                        {
                            "accountId": expected_account_id,
                            "id": "role-123",
                            "name": "StandardBasic",
                            "roleId": "5441",
                        }
                    ]
                }
            }
        }

        # Set up mocks for group lookup GET request
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        # Set up mocks for role assignment POST request
        mock_role_response = MockApi()
        mock_role_response.on("raise_for_status").return_val(None)
        mock_role_response.on("json").return_val(expected_role_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup (from get_nr_group_by_ad_name)
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Mock the POST request for role assignment
        expected_role_mutation = {
            "query": f"""
mutation {{
    authorizationManagementGrantAccess(
        grantAccessOptions: {{
            groupId: "{expected_group_id}",
            accountAccessGrants: {{accountId: {expected_account_id}, roleId: "6033"}}
        }}
    ) {{
        roles {{
            accountId
            id
            name
            roleId
        }}
    }}
}}
"""
        }
        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_role_mutation,
        ).return_val(mock_role_response)

        # Test add_nr_group_to_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.add_nr_group_to_account(
            expected_account_id, expected_group_name, expected_org_id
        )

        assert result == expected_role_response
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()
        mock_role_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_standard_fso(self):
        """Test adding a StandardFSO group to an account."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "WW New Relic Users sFSO_myapp-prod"
        expected_org_id = "67890"
        expected_group_id = "group-123-456"

        # Expected group lookup response
        expected_group_lookup_response = {
            "data": {
                "customerAdministration": {
                    "groups": {"items": [{"id": expected_group_id, "name": expected_group_name}]}
                }
            }
        }

        # Expected role assignment response
        expected_role_response = {
            "data": {
                "authorizationManagementGrantAccess": {
                    "roles": [
                        {
                            "accountId": expected_account_id,
                            "id": "role-123",
                            "name": "StandardFSO",
                            "roleId": "5441",
                        }
                    ]
                }
            }
        }

        # Set up mocks for group lookup GET request
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        # Set up mocks for role assignment POST request
        mock_role_response = MockApi()
        mock_role_response.on("raise_for_status").return_val(None)
        mock_role_response.on("json").return_val(expected_role_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup (from get_nr_group_by_ad_name)
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Mock the POST request for role assignment
        expected_role_mutation = {
            "query": f"""
mutation {{
    authorizationManagementGrantAccess(
        grantAccessOptions: {{
            groupId: "{expected_group_id}",
            accountAccessGrants: {{accountId: {expected_account_id}, roleId: "5441"}}
        }}
    ) {{
        roles {{
            accountId
            id
            name
            roleId
        }}
    }}
}}
"""
        }
        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_role_mutation,
        ).return_val(mock_role_response)

        # Test add_nr_group_to_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.add_nr_group_to_account(
            expected_account_id, expected_group_name, expected_org_id
        )

        assert result == expected_role_response
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()
        mock_role_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_advanced_fso(self):
        """Test adding an AdvancedFSO group to an account."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "WW New Relic Users aFSO_myapp-prod"
        expected_org_id = "67890"
        expected_group_id = "group-456-789"

        # Expected responses
        expected_group_lookup_response = {
            "data": {
                "customerAdministration": {
                    "groups": {"items": [{"id": expected_group_id, "name": expected_group_name}]}
                }
            }
        }

        expected_role_response = {
            "data": {
                "authorizationManagementGrantAccess": {
                    "roles": [
                        {
                            "accountId": expected_account_id,
                            "id": "role-456",
                            "name": "AdvancedFSO",
                            "roleId": "5442",
                        }
                    ]
                }
            }
        }

        # Set up mocks
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        mock_role_response = MockApi()
        mock_role_response.on("raise_for_status").return_val(None)
        mock_role_response.on("json").return_val(expected_role_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Mock the POST request for role assignment (should use role ID 5442 for AdvancedFSO)
        expected_role_mutation = {
            "query": f"""
mutation {{
    authorizationManagementGrantAccess(
        grantAccessOptions: {{
            groupId: "{expected_group_id}",
            accountAccessGrants: {{accountId: {expected_account_id}, roleId: "5442"}}
        }}
    ) {{
        roles {{
            accountId
            id
            name
            roleId
        }}
    }}
}}
"""
        }
        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_role_mutation,
        ).return_val(mock_role_response)

        # Test add_nr_group_to_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.add_nr_group_to_account(
            expected_account_id, expected_group_name, expected_org_id
        )

        assert result == expected_role_response
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()
        mock_role_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_admin(self):
        """Test adding an admin group to an account."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "WW New Relic Users POEAdmin"
        expected_org_id = "67890"
        expected_group_id = "group-admin-123"

        # Expected responses
        expected_group_lookup_response = {
            "data": {
                "customerAdministration": {
                    "groups": {"items": [{"id": expected_group_id, "name": expected_group_name}]}
                }
            }
        }

        expected_role_response = {
            "data": {
                "authorizationManagementGrantAccess": {
                    "roles": [
                        {
                            "accountId": expected_account_id,
                            "id": "role-admin",
                            "name": "All Product Admin",
                            "roleId": "1254",
                        }
                    ]
                }
            }
        }

        # Set up mocks
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        mock_role_response = MockApi()
        mock_role_response.on("raise_for_status").return_val(None)
        mock_role_response.on("json").return_val(expected_role_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Mock the POST request for role assignment (should use role ID 1254 for All Product Admin)
        expected_role_mutation = {
            "query": f"""
mutation {{
    authorizationManagementGrantAccess(
        grantAccessOptions: {{
            groupId: "{expected_group_id}",
            accountAccessGrants: {{accountId: {expected_account_id}, roleId: "1254"}}
        }}
    ) {{
        roles {{
            accountId
            id
            name
            roleId
        }}
    }}
}}
"""
        }
        mock_http_session.on(
            "post",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_role_mutation,
        ).return_val(mock_role_response)

        # Test add_nr_group_to_account()
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        result = nr.add_nr_group_to_account(
            expected_account_id, expected_group_name, expected_org_id
        )

        assert result == expected_role_response
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()
        mock_role_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_group_not_found(self):
        """Test error handling when group is not found."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "NonexistentGroup"
        expected_org_id = "67890"

        # Expected response with no groups found
        expected_group_lookup_response = {
            "data": {"customerAdministration": {"groups": {"items": []}}}
        }

        # Set up mocks
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup that returns no results
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Test add_nr_group_to_account() with group not found
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        with pytest.raises(AssertionError) as e_info:
            nr.add_nr_group_to_account(expected_account_id, expected_group_name, expected_org_id)

        assert f"Expected one group with name {expected_group_name}, found 0" in str(e_info.value)
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_invalid_group_name(self):
        """Test error handling when group name doesn't match any role pattern."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "Invalid Group Name"
        expected_org_id = "67890"
        expected_group_id = "group-invalid-123"

        # Expected group lookup response (group exists but doesn't match any role pattern)
        expected_group_lookup_response = {
            "data": {
                "customerAdministration": {
                    "groups": {"items": [{"id": expected_group_id, "name": expected_group_name}]}
                }
            }
        }

        # Set up mocks
        mock_group_response = MockApi()
        mock_group_response.on("raise_for_status").return_val(None)
        mock_group_response.on("json").return_val(expected_group_lookup_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_group_response)

        # Test add_nr_group_to_account() with invalid group name
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        with pytest.raises(ValueError) as e_info:
            nr.add_nr_group_to_account(expected_account_id, expected_group_name, expected_org_id)

        assert "No matching role found for group" in str(e_info.value)
        mock_http_session.assert_expectations()
        mock_group_response.assert_expectations()

    @pytest.mark.unit
    def test_add_nr_group_to_account_api_error_on_group_lookup(self):
        """Test error handling when API returns errors during group lookup."""
        expected_api_key = "test_api_key"
        expected_account_id = 12345
        expected_group_name = "WW New Relic Users sFSO_myapp-prod"
        expected_org_id = "67890"

        # Expected response with API errors
        expected_error_response = {"errors": [{"message": "API authentication failed"}]}

        # Set up mocks
        mock_error_response = MockApi()
        mock_error_response.on("raise_for_status").return_val(None)
        mock_error_response.on("json").return_val(expected_error_response)

        mock_http_session = MockApi()

        # Mock the GET request for group lookup that returns error
        expected_group_query = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{expected_group_name}"}}, organizationId: {{eq: "{expected_org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        mock_http_session.on(
            "get",
            NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            data=expected_group_query,
        ).return_val(mock_error_response)

        # Test add_nr_group_to_account() with API error during group lookup
        nr = NewRelic(expected_api_key, http_session=mock_http_session)
        with pytest.raises(Exception) as e_info:
            nr.add_nr_group_to_account(expected_account_id, expected_group_name, expected_org_id)

        assert "Error while searching for New Relic group:" in str(e_info.value)
        mock_http_session.assert_expectations()
        mock_error_response.assert_expectations()

    def get_mocks(
        self,
        query: str,
        output_keys: str,
        on: str = "",
        is_error: bool = 0,
        timeout: Optional[int] = None,
    ) -> Tuple[MockApi, MockApi]:
        expected_api_key = "derp"
        expected_next_cursor = ""
        expected_return_list = [
            {"name": "derp Highered", "tags": [{"key": "wallBoardProducts", "values": ["durp"]}]}
        ]
        expected_full_query = {
            "query": f"{{actor {{{query} {{results{expected_next_cursor} {{entities {{{on} {output_keys}}}nextCursor}}}}}}}}"
        }
        expected_resp_json = {
            "data": {"actor": {"entitySearch": {"results": {"entities": expected_return_list}}}},
            "nextCursor": None,
        }
        mock_request = MockApi()
        setattr(mock_request, "url", NEWRELIC_GRAPHQL_URL)
        setattr(mock_request, "method", "POST")
        setattr(mock_request, "headers", {})
        mock_resp = MockApi()
        setattr(mock_resp, "request", mock_request)
        if is_error:
            expected_resp_json["errors"] = [{"message": "test error"}]
            mock_resp.on("raise_for_status").raise_exception(Exception("test error"))
        else:
            mock_resp.on("raise_for_status").return_val(None)
        mock_resp.on("json").return_val(expected_resp_json)
        mock_http_session = MockApi()
        mock_http_session.on(
            "post",
            url=NEWRELIC_GRAPHQL_URL,
            headers={"API-Key": expected_api_key},
            json=expected_full_query,
            timeout=timeout,
        ).return_val(mock_resp)
        return mock_http_session, mock_resp
