import logging
from typing import Any, Optional

import requests

from poe_new_relic import get_logger, log_it


class ExperimentalClient:
    def __init__(
        self,
        username: str,
        password: str,
        session: Optional[requests.Session] = None,
        logger: Optional[logging.Logger] = None,
    ):
        self.__username: str = username
        self.__password: str = password
        self._logger: logging.Logger = logger or get_logger()
        self._http_session: requests.Session = session or requests.Session()
        self._http_session.headers["x-requested-with"] = "XMLHttpRequest"

    def __enter__(self) -> "ExperimentalClient":
        response: requests.Response = self._http_session.post(
            "https://login.newrelic.com/login",
            data={"login[email]": self.__username, "login[password]": self.__password},
            allow_redirects=False,
        )
        response.raise_for_status()

        if getattr(response.next, "url", "") == "https://login.newrelic.com/limit_active_sessions":
            raise TooManySessionsError(self.__username)

        if not response.cookies.get("login_service_login_newrelic_com_tokens"):
            self._logger.warning(
                "Expected to find login_service_login_newrelic_com_tokens cookie after logging in as %s. Future requests may fail, or New Relic may have changed their login flow.",
                self.__username,
            )  # Requests automatically adds all cookies to the session, so even if this cookie isn't present, subsequent requests may still work if they just change the name of the cookie.
        else:
            self._logger.info("Successfully logged in as %s", self.__username)

        return self

    def __exit__(self, exc_type, exc_value, traceback) -> bool | None:
        """Logout of the account's current session."""
        response: requests.Response = self._http_session.get(
            "https://login.newrelic.com/logout",
            allow_redirects=False,
        )
        response.raise_for_status()
        return True

    @log_it
    def decline_user_upgrade_request(self, request_id: str) -> dict:
        """
        Decline a user upgrade request in New Relic with a standardized rejection message.
        This method is intended for use by administrators to manage user upgrade requests.

        This method uses the reviewPendingRequest GraphQL mutation to deny a user's request
        for elevated permissions or tier upgrades. The rejection includes a standard message
        directing users to the proper myAccess workflow for requesting New Relic access.

        Parameters
        ----------
        request_id : str
            The unique identifier of the user upgrade request to decline. This can be obtained from
            a call to get_pending_user_upgrade_requests() from the standard NewRelic client from the
            api module.

        Returns
        -------
        dict
            A dictionary containing the updated request information after denial with the following keys:
            - 'id' : str
                The unique identifier of the request
            - 'status' : str
                The status of the request, will be "DENIED" after successful execution
            - 'requestType' : str
                The type of request, typically "TIER_UPGRADE"
            - 'requestMessage' : str or None
                Any message associated with the original request, may be None

            {
                "id": "310355",
                "status": "DENIED",
                "requestType": "TIER_UPGRADE",
                "requestMessage": None
            }

        Raises
        ------
        requests.HTTPError
            If the HTTP request to the New Relic API fails.
        Exception
            If the New Relic API returns errors in the response.

        Notes
        -----
        The method posts to the New Relic user GraphQL endpoint at:
        https://user-graphql.service.newrelic.com/public/graphql

        Examples
        --------
        Decline a specific user upgrade request:

        >>> with ExperimentalClient("admin@example.com", "password") as client:
        ...     result = client.decline_user_upgrade_request("310355")
        ...     print(f"Request {result['id']} status: {result['status']}")
        Request 310355 status: DENIED
        """
        mutation: str = f"""
mutation {{
  reviewPendingRequest(
    input: {{
      id: "{request_id}",
      status: DENIED,
      reviewMessage: "New Relic access is managed through Active Directory and myAccess.\nPlease request the appropriate role for the requested access.\nYou can learn more about it here: https://pearsoneducationinc.sharepoint.com/sites/GRPTOToolsSupport-SETs-/SitePages/New-Relic-User-Access-Request---myAccess.aspx"
    }}
  ) {{
    request {{
      id
      status
      requestType
      requestMessage
    }}
    errors {{
      message
    }}
  }}
}}
        """
        response: requests.Response = self._http_session.post(
            "https://user-graphql.service.newrelic.com/public/graphql", json={"query": mutation}
        )
        response.raise_for_status()
        result = response.json()
        if result.get("errors"):
            raise Exception("Error declining user upgrade request:", result["errors"])
        return result["data"]["reviewPendingRequest"]["request"]


class TooManySessionsError(Exception):
    """Exception raised when there are too many active sessions for a user."""

    def __init__(self, username: str):
        super().__init__(
            f"Too many active sessions for user {username}. You must manually sign in as {username} and end some sessions before using this client."
        )
