import logging
import os
from typing import Any, Literal, Optional

import requests

from poe_new_relic import get_logger, log_it

NERDGRAPH_MAX_TIMEOUT_MS = 30 * 60 * 1000  # 30 mins
NEWRELIC_GRAPHQL_URL = os.environ.get("NERDGRAPH_URL", "https://api.newrelic.com/graphql")


class NewRelic:
    def __init__(
        self,
        api_key: str,
        base_url: Optional[str] = None,
        http_session: requests.Session = requests.Session(),
        logger: Optional[logging.Logger] = None,
    ):
        self._http_session = http_session
        self.__logger = logger or get_logger()
        self._base_url = base_url or NEWRELIC_GRAPHQL_URL
        self._headers = {
            "API-Key": api_key,
        }

    @log_it
    def execute_nerdgraph_query(
        self,
        query: str,
        variables: Optional[str] = None,
        timeout_ms: Optional[int] = None,
    ) -> dict[str, Any]:
        """Execute a NerdGraph query.

        Parameters
        ----------
        query : str
            The NerdGraph query to execute
        variables : str
            (optional) Variables to augment the NerdGraph query
        timeout_ms : int
            (optional) How long (in milliseconds) the query should run before NerdGraph gives up.
            If unset, NerdGraph uses the default of 5 seconds.

        Returns
        -------
        dict[str, Any]
            The NerdGraph response as a dictionary.
        """
        if timeout_ms is not None:
            assert (
                0 < timeout_ms <= NERDGRAPH_MAX_TIMEOUT_MS
            ), f"NerdGraph query timeout ms must be in the range [0, {NERDGRAPH_MAX_TIMEOUT_MS})"

        try:
            full_query = {"query": query}
            if variables:
                full_query["variables"] = variables

            # Pass timeout as a separate argument, not in the JSON payload
            resp = self._http_session.post(
                url=self._base_url, headers=self._headers, json=full_query, timeout=timeout_ms
            )
            resp.raise_for_status()
            return resp.json()
        except Exception as e:
            self.__logger.error(f"Failed to execute NerdGraph query with error: {e}")
            raise e

    def nerdgraph_nrql_search(
        self,
        account_id: int,
        nrql: Optional[str] = None,
        label: str = "entities",
        nrqls: Optional[dict[str, str]] = None,
        timeout_ms: int = 10,  # Defaults to 10 seconds
    ) -> dict[str, dict]:
        """Run a NerdGraph NRQL search by either specifying a single NRQL query to run or a map of
        label -> NRQL query to run (allows running multiple NRQL queries at once).

        Parameters
        ----------
        account_id : int
            The NewRelic account to run the NRQL query in.
        nrql : str
            (conditional) The NRQL query string to run. Must be set if the 'nrqls' parameter isn't.
        label : str
            (optional) The label given to the NRQL query specified by the 'nrql' parameter.
            Does not apply to the 'nrqls' parameter.
        nrqls : dict[str, str]
            (conditional) A mapping of label -> NRQL query allowing for running multiple NRQL
            queries at once. E.g.

            {
                "apmAppNames": "SELECT uniques(appName) FROM Transaction SINCE 1 hour ago",
                "browserAppNames": "SELECT uniques(appName) FROM BrowserInteraction SINCE 1 hour ago"
            }
        timeout_ms : int
            (optional) How long (in milliseconds) the query should run before NerdGraph gives up.
            If unset, NerdGraph uses the default of 5 seconds.

        Returns
        -------
        dict[str, dict]
            The NerdGraph response as a dictionary with the form:

            {
                "exampleLabel1": {
                    "results": [<list of result dicts>]
                },
                "exampleLabel2": {
                    "results": [<list of result dicts>]
                }
            }
        """
        assert bool(nrql) != bool(
            nrqls
        ), "To run a NerdGraph NRQL search, must specify exactly one of nrql or nrqls"
        if nrql:
            nrqls = {label: nrql}

        self.__logger.info(f"Running NerdGraph NRQL search with queries: {nrqls}")
        try:
            nrqls = ",".join(
                [f'{k.strip()}: nrql(query: "{v}") {{results}}' for k, v in nrqls.items()]
            )
            query = f"{{actor {{account(id: {account_id}) {{{nrqls}}}}}}}"
            resp_json = self.execute_nerdgraph_query(query, timeout_ms=timeout_ms)
            return resp_json.get("data", {}).get("actor", {}).get("account", {})
        except Exception as e:
            self.__logger.error(f"Failed to run NRQL query with error: {e}")
            raise e

    def nerdgraph_entitysearch(
        self,
        query: str,
        output_keys: str,
        on: str = "",
        timeout_ms: Optional[int] = None,
    ) -> dict:
        """_summary_
            query New Relic's Nerdgraph api, https://api.newrelic.com/graphiql
        Args:
            query (str): the SELECT section of the NRQL query
            on (str): appears to be a sub filtering of what to return and contains proprietary return values for what you pick
            output_keys (str): the keys to return values for

        Returns:
            dict: dynamic results defined by the output_keys arg
        """
        self.__logger.info(f"Running entity search with query: {query}")
        entities, next_cursor = [], ""
        while True:
            try:
                if next_cursor:
                    next_cursor = f'(cursor: "{next_cursor}")'
                # TODO: allow for multiple 'on' values to be used and built out in the query string like in the nerdgraph explorer
                resp_json = self.execute_nerdgraph_query(
                    query=f"{{actor {{{query} {{results{next_cursor} {{entities {{{on} {output_keys}}}nextCursor}}}}}}}}",
                    timeout_ms=timeout_ms,
                )
                results = resp_json["data"]["actor"]["entitySearch"]["results"]
                entities.extend(results.get("entities"))
                self.__logger.debug(f"Total entities: {len(entities)}")
                next_cursor = results.get("nextCursor", None)
                if not next_cursor:
                    self.__logger.info(
                        f"Completed entity search, found {len(entities)} entity/entities"
                    )
                    break
            except Exception as e:
                self.__logger.error(f"Failed to run entity search with error: {e}")
                raise e
        return entities

    def get_synths(self) -> list:
        """_summary_
            cursors over Nerdgraph's entitySearch query results to return every synthetic
        Raises:
            e: logs the response back from new relic regarding the query attempt and raises the actual error
        Returns:
            list: the combined cursored results of all of the synthetics
        """
        entity_search = "entitySearch(query: \"domain = 'SYNTH' AND type = 'MONITOR'\")"
        on = "... on SyntheticMonitorEntityOutline"
        output_keys = "{guid name tags {key values} monitorId}"
        resp = self.nerdgraph_entitysearch(entity_search, output_keys, on)
        return resp

    def get_monitorid_and_guids_synths_by_tag(self, tag: str) -> dict:
        """_summary_
            groups monitorIds and guids by tag in an array
        Args:
            tag (str): tag of an entity

        Returns:
            dict: containing per tag grouping of an array of monitorIds and array of guids
        """
        wb_tagged = []
        all_synths = self.get_synths()
        # get only synthetics having said tag
        wb_tagged.extend(
            [
                i
                for e in all_synths
                for i in [",".join(t["values"]) for t in e.get("tags") if t["key"] == tag]
            ]
        )
        self.__logger.info(f"Found {len(wb_tagged)} entities with tag {tag}")
        # geting distinct set of tags
        set_tags = set(wb_tagged)
        # reducing the  keys down grabbing monitorIds and guids per tag
        panel_entities = [
            {t: {"monitorId": e.get("monitorId"), "guid": e.get("guid")}}
            for e in all_synths
            for t in set_tags
            for l in [w for w in e.get("tags") if t in w["values"]]
        ]
        # grouping tags to array of dicts, {tag:[{monitorId:value, guid:value}]}
        panel_entities = {
            k: [d.get(k) for d in panel_entities if k in d] for k in set().union(*panel_entities)
        }
        # creating arrays per monitorId and per guid per tag, {tag:[{monitorId:[values]},{guid:[values]}]} TODO change array of dicts for tags to single dict with the keys
        panel_entities = {
            g: [
                {
                    k: [t.get(k) for t in panel_entities.get(g) if k in t]
                    for x in panel_entities.get(g)
                }
                for k in set().union(*panel_entities.get(g))
            ]
            for g in set_tags
            if panel_entities.get(g)
        }
        return panel_entities

    def get_scorecard_performance_dashboards(self) -> list:
        """_summary_
            returns the raw array of entities of dashboards filtered on name like Performance and Scorecard
        Raises:
            e: logs the response back from new relic regarding the query attempt and raises the actual error

        Returns:
            list: array of the entities built out from the results of the cursor/paginator
        """
        entity_search = "entitySearch(query: \"((name LIKE 'Scorecard -%') OR (name LIKE 'Performance -%')) AND (domain = 'VIZ' AND type = 'DASHBOARD')\")"
        on = "... on DashboardEntityOutline"
        output_keys = "{guid name updatedAt accountId permalink createdAt}"
        resp = self.nerdgraph_entitysearch(entity_search, output_keys, on)
        main_dashboards = [
            x
            for x in resp
            if x
            for k, v in x.items()
            if k == "name" and v.split()[-1] == "Highered"
        ]
        self.__logger.info(
            f"Found {len(main_dashboards)} Scorecard and Performance dashboards found"
        )
        return main_dashboards

    def create_account(
        self,
        name: str,
        environment: str,
        lob_suffix: str,
        account_type: Literal["standard", "private", "hsm", "fedramp"],
        org_id: str,
        region: str = "us01",
    ) -> dict[str, str | int]:
        """
        Create a new New Relic account within an organization using the NerdGraph API.

        This method creates a managed account in the specified New Relic organization with a
        standardized naming convention based on the provided parameters.
        The account name format differs based on the account type.

        Parameters
        ----------
        name : str
            The base name for the New Relic account. This will be used as the prefix
            in the generated account name.
        environment : str
            The environment designation for the account (e.g., 'prd', 'dev').
            This will be included in the generated account name.
        lob_suffix : str
            The line of business suffix code. Must be one of: 'asmq', 'ellx', 'gptx',
            'strg', 'vlxx', 'wfsx'. This will be used as the suffix in the account name.
        account_type : Literal["standard", "private", "hsm", "fedramp"]
            The type of New Relic account to create. This determines the naming convention:
            - "standard": Standard account with no special changes
            - "private": Private account with "_SECURE" added to the name
            - "hsm": High Security Mode account with "_HSM" added to the name
            - "fedramp": FedRAMP compliant account with "_FR" added to the name
        org_id : str
            The New Relic organization ID where the account should be created.
        region : str, optional
            The New Relic region code where the account should be created. Must be
            one of: 'us01', 'eu01'. Default is 'us01'.

        Returns
        -------
        dict[str, str | int]
            A dictionary containing the created account information with the following keys:
            - 'name' : str
                The full name of the created account
            - 'id' : int
                The unique identifier of the created account

            {
                "id": 6719983,
                "name": "panext_SECURE-prd_asmq"
            }

        Raises
        ------
        AssertionError
            If `lob_suffix` is not in the list of valid LOB suffixes.
            If `region` is not in the list of valid regions.
            If `account_type` is not in the list of valid account types.
        requests.HTTPError
            If the HTTP request to the New Relic API fails.
        Exception
            If the New Relic API returns errors in the response.

        Notes
        -----
        Account naming conventions:
        - Standard accounts: "{name}-{environment}_{lob_suffix}"
        - Private accounts:  "{name}_SECURE-{environment}_{lob_suffix}"
        - HSM accounts:      "{name}_HSM-{environment}_{lob_suffix}"
        - FedRAMP accounts:  "{name}_FR-{environment}_{lob_suffix}"

        Valid LOB suffixes: asmq, ellx, gptx, strg, vlxx, wfsx
        Valid regions: us01 (US), eu01 (Europe - Not currently used by Pearson)
        Valid account types: standard, private, hsm, fedramp

        Examples
        --------
        Create a standard production account:

        >>> client = NewRelic(api_key="your_key")
        >>> result = client.create_account(
        ...     name="MyApp",
        ...     environment="prod",
        ...     lob_suffix="gptx",
        ...     account_type="standard",
        ...     org_id="12345"
        ... )
        >>> print(result)
        {'name': 'MyApp-prod_gptx', 'id': 67890}

        Create a private development account:

        >>> result = client.create_account(
        ...     name="TestApp",
        ...     environment="dev",
        ...     lob_suffix="strg",
        ...     account_type="private",
        ...     org_id="12345"
        ... )
        >>> print(result)
        {'name': 'TestApp_SECURE-dev_strg', 'id': 67891}

        Create an HSM production account:

        >>> result = client.create_account(
        ...     name="SecureApp",
        ...     environment="prod",
        ...     lob_suffix="asmq",
        ...     account_type="hsm",
        ...     org_id="12345"
        ... )
        >>> print(result)
        {'name': 'SecureApp_HSM-prod_asmq', 'id': 67892}
        """
        valid_lobs: list[str] = ["asmq", "ellx", "gptx", "strg", "vlxx", "wfsx"]
        valid_regions: list[str] = ["us01", "eu01"]
        valid_account_types: list[str] = ["standard", "private", "hsm", "fedramp"]

        assert lob_suffix in valid_lobs, f"Invalid lob_suffix. Must be one of {valid_lobs}"
        assert region in valid_regions, f"Invalid region. Must be one of {valid_regions}"
        assert (
            account_type in valid_account_types
        ), f"Invalid account_type. Must be one of {valid_account_types}"

        if account_type == "standard":
            account_name: str = f"{name}-{environment}_{lob_suffix}"
        elif account_type == "private":
            account_name: str = f"{name}_SECURE-{environment}_{lob_suffix}"
        elif account_type == "hsm":
            account_name: str = f"{name}_HSM-{environment}_{lob_suffix}"
        elif account_type == "fedramp":
            account_name: str = f"{name}_FR-{environment}_{lob_suffix}"

        mutation: str = f"""
mutation {{
    accountManagementCreateAccount(
        managedAccount: {{
            regionCode: "{region}",
            organizationId: "{org_id}",
            name: "{account_name}"}}
    ) {{
        managedAccount {{
            name
            id
        }}
    }}
}}
"""

        response: requests.Response = self._http_session.post(
            self._base_url, headers=self._headers, json={"query": mutation}
        )
        response.raise_for_status()
        result = response.json()
        if result.get("errors"):
            raise Exception("Error while submitting account creation request:", result["errors"])
        return result["data"]["accountManagementCreateAccount"]["managedAccount"]

    def get_nr_group_by_ad_name(self, name: str, org_id: str) -> dict[str, str]:
        """
        Retrieve a New Relic group by Active Directory name using the NerdGraph API.

        This method searches for a specific group within a New Relic organization using the group's
        Active Directory name. It expects to find exactly one matching group.

        Parameters
        ----------
        name : str
            The Active Directory name of the New Relic group to search for.
            This should be the exact name as it appears in New Relic
        org_id : str
            The New Relic organization ID where the group should be found.

        Returns
        -------
        dict[str, str]
            A dictionary containing the group information with the following keys:
            - 'id' : str
                The unique identifier of the group
            - 'name' : str
                The name of the group as stored in New Relic

            {
                "id": "f1ad2569-eb52-474f-9911-689fb654c5d7",
                "name": "WW New Relic Users aFSO_oscar-prd"
            }

        Raises
        ------
        requests.HTTPError
            If the HTTP request to the New Relic API fails.
        AssertionError
            If the number of groups found is not exactly 1.
            This occurs when no groups are found or multiple groups match the name.
        Exception
            If the New Relic API returns errors in the response.

        Notes
        -----
        This method uses New Relic's customerAdministration API to query groups by name and
        organization ID. The search is not case-sensitive and requires an exact match on the name.

        Examples
        --------
        Retrieve a specific group by name:

        >>> client = NewRelic(api_key="your_key")
        >>> group = client.get_nr_group_by_ad_name(
        ...     name="WW New Relic Users aFSO_adam-prd",
        ...     org_id="12345"
        ... )
        >>> print(group)
        {'id': '67890', 'name': 'WW New Relic Users aFSO_adam-prd'}
        """
        query: str = f"""
{{
    customerAdministration {{
        groups(
            filter: {{name: {{eq: "{name}"}}, organizationId: {{eq: "{org_id}"}}}}
        ) {{
            items {{
                id
                name
            }}
        }}
    }}
}}
"""
        response: requests.Response = self._http_session.get(
            self._base_url, headers=self._headers, data=query
        )
        response.raise_for_status()
        result = response.json()
        if result.get("errors"):
            raise Exception("Error while searching for New Relic group:", result["errors"])
        groups: list[dict[str, str]] = result["data"]["customerAdministration"]["groups"]["items"]
        assert len(groups) == 1, f"Expected one group with name {name}, found {len(groups)}"
        return groups[0]

    def add_nr_group_to_account(self, account_id: int, group_name: str, org_id: str):
        """
        Add a New Relic group to an account with the appropriate role assignment.

        This method assigns a New Relic group to a specific account. The role is automatically
        determined based on the group name pattern, following organizational naming conventions
        for different access levels.

        Parameters
        ----------
        account_id : int
            The unique identifier of the New Relic account where the group should be granted access.
        group_name : str
            The name of the New Relic group to add to the account. The group name determines the
            role assignment based on naming patterns:
            - Groups containing 'sFSO' or 'StandardFSO' get the StandardFSO role
            - Groups containing 'aFSO' or 'AdvancedFSO' get the AdvancedFSO role
            - Groups containing 'sBasic' or 'StandardBasic' get the StandardBasic role
            - Admin groups get the All Product Admin role
            - The 'WW New Relic Users TDMViewer' group gets the Viewer-all role
        org_id : str
            The New Relic organization ID that contains both the group and account.

        Returns
        -------
        dict[str, Any]
            The response from the New Relic API containing role assignment information.
            Structure includes details about the granted roles and their properties.

        Raises
        ------
        requests.HTTPError
            If the HTTP request to the New Relic API fails.
        AssertionError
            If the specified group is not found in the organization.
        ValueError
            If the group name does not match any of the expected patterns for role assignment.
        Exception
            If the New Relic API returns errors in the response during group lookup
            or role assignment.

        Notes
        -----
        Role assignments based on group name patterns:

        - **StandardFSO (5441)**: Groups with 'sFSO' or 'StandardFSO' in name
        - **AdvancedFSO (5442)**: Groups with 'aFSO', 'AdvancedFSO', or 'API_all'
        - **StandardBasic (6033)**: Groups with 'sBasic' or 'StandardBasic' in name
        - **All Product Admin (1254)**: Admin groups ('WW New Relic Users POEAdmin', 'Admin', 'API_all')
        - **Viewer-all (40979)**: The 'WW New Relic Users TDMViewer' group

        Examples
        --------
        Add a StandardFSO group to an account:

        >>> client = NewRelic(api_key="your_key")
        >>> response = client.add_nr_group_to_account(
        ...     account_id=12345,
        ...     group_name="WW New Relic Users sFSO_myapp-prod",
        ...     org_id="7890"
        ... )
        """
        group_id: str = self.get_nr_group_by_ad_name(group_name, org_id)["id"]
        self.__logger.info("Group name: %s, group ID: %s", group_name, group_id)

        role_id: Literal["", "5441", "5442", "6033", "1254", "40979"] = ""
        if "sFSO" in group_name or "StandardFSO" in group_name:
            role_id = "5441"
            self.__logger.info("Using StandardFSO role (ID: %s) for group %s", role_id, group_name)
        elif "aFSO" in group_name or "AdvancedFSO" in group_name or group_name == "API_all":
            role_id = "5442"
            self.__logger.info("Using AdvancedFSO role (ID: %s) for group %s", role_id, group_name)
        elif "sBasic" in group_name or "StandardBasic" in group_name:
            role_id = "6033"
            self.__logger.info(
                "Using StandardBasic role (ID: %s) for group %s", role_id, group_name
            )
        elif group_name in ["WW New Relic Users POEAdmin", "Admin", "API_all"]:  # Admin groups
            role_id = "1254"
            self.__logger.info(
                "Using All Product Admin role (ID: %s) for group %s", role_id, group_name
            )
        elif group_name == "WW New Relic Users TDMViewer":
            role_id = "40979"
        else:
            raise ValueError("No matching role found for group %s", group_name)

        mutation: str = f"""
mutation {{
    authorizationManagementGrantAccess(
        grantAccessOptions: {{
            groupId: "{group_id}",
            accountAccessGrants: {{accountId: {account_id}, roleId: "{role_id}"}}
        }}
    ) {{
        roles {{
            accountId
            id
            name
            roleId
        }}
    }}
}}
"""

        response: requests.Response = self._http_session.post(
            self._base_url, headers=self._headers, json={"query": mutation}
        )
        response.raise_for_status()
        result = response.json()
        if result.get("errors"):
            raise Exception(f"Error while adding group {group_name} to account:", result["errors"])
        return result

    def get_pending_user_upgrade_requests(
        self, auth_domain_id: Optional[str] = "5db7da08-fd35-4913-b911-16fa965fdb34"
    ) -> list[dict]:
        """
        Retrieve users with pending upgrade requests from a New Relic authentication domain.
        This is used to identify users requesting elevated permissions or access levels.

        Parameters
        ----------
        auth_domain_id : str, optional
            The unique identifier of the New Relic authentication domain to search within.
            Default is "5db7da08-fd35-4913-b911-16fa965fdb34" which corresponds to the
            AzureSCIM authentication domain used by the Pearson New Relic organization.

        Returns
        -------
        list[dict]
            A list of user dictionaries containing information about users with pending
            upgrade requests.
            [
                {
                    'email': 'test.user@pearson.com',
                    'id': '1007032335', # Unique user ID
                    'pendingUpgradeRequest': {
                        'id': '310282', # Unique request ID
                        'requestedUserType': {'id': '0', 'name': 'Full platform'},
                        'type': {'id': '1', 'name': 'Basic'}  # Current user type
                    }
                }
            ]

        Raises
        ------
        requests.HTTPError
            If the HTTP request to the New Relic API fails.
        Exception
            If the New Relic API returns errors in the response.

        Notes
        -----
        The default auth_domain_id corresponds to the AzureSCIM domain used by the Pearson org.
        Different authentication domains may be used for other identity providers or configurations.

        Examples
        --------
        Get pending upgrade requests using the default authentication domain:
        >>> client = NewRelic(api_key="your_key")
        >>> pending_users = client.get_pending_user_upgrade_requests()
        >>> for user in pending_users:
        ...     print(f"User {user['email']} requesting {user['pendingUpgradeRequest']['requestedUserType']['name']}")
        """
        query: str = f"""
{{
    customerAdministration {{
        users(
            filter: {{authenticationDomainId: {{eq: "{auth_domain_id}"}},
                    pendingUpgradeRequest: {{exists: true}}}}
        ) {{
            items {{
                type {{
                    name
                    id
                }}
                pendingUpgradeRequest {{
                    id
                    requestedUserType {{
                    id
                    name
                    }}
                }}
            id
            email
            }}
        }}
    }}
}}
"""
        response: requests.Response = self._http_session.post(
            self._base_url, headers=self._headers, data=query
        )
        response.raise_for_status()
        result = response.json()
        if result.get("errors"):
            raise Exception("Error getting pending upgrade requests:", result["errors"])
        users: list[dict] = result["data"]["customerAdministration"]["users"]["items"]
        self.__logger.info(f"Found {len(users)} users with pending upgrade requests")
        return users
